// This program solves the time dependent equations for
//  the simple model of exponential decay. There are 2N-1 "continuum"
//  states with equal coupling to the one state that can decay
//
// This program uses the leapfrog algorithm
// psi(t+dt) = psi(t-dt) - 2 i H(t) psi(t)
//

#include <stdio.h>   // standard IO
#include <string.h>  // string manipulations
#include <math.h>    // alternate math functions
#include <complex>   // package for doing complex manipulations
#include <cmath>

// define the type called couble complex
using namespace std ;
typedef complex<double> dcx;

//main program
int main()
{
const int nabv = 2000 ;
dcx ceve[2*nabv], codd[2*nabv] ;
long int numtim, nmod ;
double gam, delw, coe, omeg0, dt, parm, tfin, tim, fun, p1, p2, w ;

//define the complex number i
const complex <double> icom (0.0,1.0) ;
//define the number pi
const double pi = double(2)*asin(double(1)) ;

//the output will be in whatever is defined here
FILE *output ; output = fopen("exp_decay_1E9_2E7_2000_cpp.txt","w") ;
FILE *output2; output2= fopen("exp_edist_1E9_2E7_2000_cpp.txt","w") ;

// the coupling will be put in as a decay rate and Delta_omega
// gam = Gamma = the decay rate of the continuum limit
gam = 1.e9 ;
// delw = Delta_omega = Delta_E/hbar
delw = gam*0.02 ;
// now define the coupling term
coe = sqrt(gam/2.0/pi) ;
// set the state energy E_0 = hbar omeg0
omeg0 = 0.0 ;
// set the time step (you should check convergence with respect to dt)
dt = 0.025/(nabv*delw) ;
// the combination coe*sqrt(delw)*dt occurs alot so define it to be parm
parm = coe*sqrt(delw)*dt ;
// define the final time (tfin)
tfin = 15.0/gam ;
// define the number of steps in the numerical integration
numtim = (tfin/dt) ;
      
// initialize the coefficients in the two state calculation
ceve[0] = dcx(1.0,0.0) ;
ceve[1] = dcx(0.0,0.0) ;
// these initial conditions for codd assume the E-field is 0 at t=0
codd[0] = ceve[0] ;
codd[1] = ceve[1] ;
// initialize the ceve=c(0) and codd(dt) by using the 2nd order Runge-Kutta alg.
// initialize the coefficients at t = dt/2
ceve[0] = dcx(1.0,0.0) ;
codd[0] = ceve[0] ;
 for(long int j = 1 ; j < 2*nabv ; j++)
 {
 ceve[j] = -icom*parm*0.5 ;
 codd[j] = dcx(0.0,0.0) ;
 }
ceve[0] = ceve[0] - icom*0.5*omeg0*ceve[0] ;
// at this point ceve = c(dt/2) and codd = c(0)
codd[0] = codd[0] - icom*dt*omeg0*ceve[0] ;
 for(long int j = 1 ; j < 2*nabv ; j++)
 {
 w = (j-nabv)*delw ;
 codd[0] = codd[0] - icom*parm*ceve[j] ;
 codd[j] = codd[j] - icom*parm*ceve[0] - icom*w*dt*ceve[j] ;
 ceve[j] = dcx(0.0,0.0) ;
 }
ceve[0] = dcx(1.0,0.0) ;
// now the ceve = c(0) and codd = c(dt)
    
// nmod is roughly how many steps to skip so the number of output points
//    are the denominator
nmod = numtim/2000 ;
      
// the next loop is the time steps
//  because of how the leapfrog algorithm works, the times are stepped in 2's
 for(long int itim = 2 ; itim <= numtim ; itim+= 2)
 {
// define the time for step j
 tim = double(itim-1)*dt ;
// step the psi(t)
 ceve[0] = ceve[0] - icom*(2*dt*omeg0)*codd[0] ;
 for(long int j = 1 ; j < 2*nabv ; j++)
 {
 w = (j-nabv)*delw ;
 ceve[0] = ceve[0] - icom*(2*parm)*codd[j] ;
 ceve[j] = ceve[j] - icom*(2*parm)*codd[0] - icom*(2*w*dt)*codd[j] ;
 }

 tim = double(itim)*dt ;
// step the psi(t+dt)
 codd[0] = codd[0] - icom*(2*dt*omeg0)*ceve[0] ;
 for(long int j = 1 ; j < 2*nabv ; j++)
 {
 w = (j-nabv)*delw ;
 codd[0] = codd[0] - icom*(2*parm)*ceve[j] ;
 codd[j] = codd[j] - icom*(2*parm)*ceve[0] - icom*(2*w*dt)*ceve[j] ;
 }

// every nmod steps output the populations
  if(itim%nmod <= 1)
  {
// write the time in microsecs, P_1, P_2, P_1+P_2 to the screen
  p1 = norm(ceve[0]) ;
  p2 = 0.0 ;
   for(long int j = 1 ; j < 2*nabv ; j++) p2 += norm(ceve[j]) ;
  printf("%15.7E %15.7E %15.7E %15.7E\n",tim*1.e6,p1,p2,p1+p2) ;
// write the timein microsecs, P_1, P_2, P_1+P_2 to the output file
  fprintf(output,"%15.7E %15.7E %15.7E %15.7E\n",tim*1.e9,p1,p2,p1+p2) ;
  }
 }
// end of time loop
 fclose(output) ;

// output the distribution of final states
 for(long int j = 1 ; j < 2*nabv ; j++)
 {
 w = (j-nabv)*delw ;
 fprintf(output2,"%15.7E %15.7E %15.7E\n",w,norm(ceve[j]),
           (gam*delw/2.0/pi)/(0.25*gam*gam+(w-omeg0)*(w-omeg0))) ;
 }
return 0 ;
}
